package furny.ga.operators;

import furny.ga.FurnEntry;
import furny.ga.FurnLayoutIndividual;
import furny.ga.PseudoSpace;
import furny.ga.RoomVector;
import furny.ga.util.FurnLayoutIOUtil;
import furny.ga.util.FurnitureUtil;
import ga.core.GA;
import ga.core.goperators.IMutationOp;
import ga.core.goperators.ProbabilityOp;
import ga.core.validation.GAContext;
import ga.view.interfaces.IPhenotypeSpace;

/**
 * This mutation operator is an advanced uniform mutation for furniture entries.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
public class FurnLayoutMutationOp extends ProbabilityOp implements
    IMutationOp<FurnLayoutIndividual> {

  private static final int MAX_ROTATION = 3;

  private final int wTranslate = 50;

  // TODO how much mutation? there is a need to set the mutation rate for
  // entries
  //
  // pMove and pExchange are needed

  /**
   * Creates the operator with the given mutation probability.
   * 
   * @param pMutate
   *          Mutation probability.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public FurnLayoutMutationOp(final int pMutate) {
    super(pMutate);
  }

  @Override
  public FurnLayoutIndividual mutate(final FurnLayoutIndividual individual,
      final GAContext context) {
    if (doOperate()) {
      final FurnLayoutIndividual newInd = individual.clone();

      for (final FurnEntry entry : newInd.getFurnitures()) {
        if (getRandom().nextBoolean()) {
          // mutate the translation and rotation

          RoomVector vec = entry.getVector();
          final int translation = getRandom().nextInt(wTranslate)
              - (wTranslate / 2);

          // mutation on all genes with pMut=0.5 (Uniform)
          if (getRandom().nextBoolean()) {
            vec = vec.getTranslatedInstance(translation, 0);
          }

          if (getRandom().nextBoolean()) {
            vec = vec.getTranslatedInstance(0, translation);
          }

          if (getRandom().nextBoolean()) {
            vec = vec.getRotatedInstance(getRandom().nextInt(MAX_ROTATION) - 1);
          }

          entry.setVector(vec);
        } else {

          final Object o = context.get(GA.KEY_VALIDATION_SPACE);

          if (o != null && o instanceof IPhenotypeSpace) {
            entry.setFurniture(FurnitureUtil
                .getOtherFurnitureBySimilarityRoulette(entry.getFurniture()));

          } else {
            throw new RuntimeException(
                "Phenotype space not found in the ga context");
          }
        }
      }

      return newInd;
    } else {
      return individual;
    }
  }

  /**
   * Main method for testing.
   * 
   * @param args
   *          No arguments required.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public static void main(final String[] args) {
    final GAContext context = new GAContext();
    context.put(GA.KEY_VALIDATION_SPACE, new PseudoSpace(10f, 10f));

    final FurnLayoutIndividual ind = new FurnLayoutIndividual(context);
    ind.initRandomly();

    final FurnLayoutMutationOp crossover = new FurnLayoutMutationOp(100);

    System.out.println("\n BEFORE\n");
    System.out.println(FurnLayoutIOUtil.printSimpleGenotype(ind
        .getSimpleGenotype()));

    final FurnLayoutIndividual newInd = crossover.mutate(ind, context);

    System.out.println("\n AFTER\n");
    System.out.println(FurnLayoutIOUtil.printSimpleGenotype(newInd
        .getSimpleGenotype()));
  }
}
